using System;
using Server;
using Server.Gumps;
using Server.Network;
using Server.Menus;
using Server.Menus.Questions;
using Server.Accounting;
using Server.Multis;
using Server.Mobiles;

namespace Server.Engines.Help
{
	public class ContainedMenu : QuestionMenu
	{
		private Mobile m_From;

		public ContainedMenu( Mobile from ) : base( "You already have an open help request. We will have someone assist you as soon as possible.  What would you like to do?", new string[]{ "Leave my old help request like it is.", "Remove my help request from the queue." } )
		{
			m_From = from;
		}

		public override void OnCancel( NetState state )
		{
			m_From.SendLocalizedMessage( 1005306, "", 0x35 ); // Help request unchanged.
		}

		public override void OnResponse( NetState state, int index )
		{
			if ( index == 0 )
			{
				m_From.SendLocalizedMessage( 1005306, "", 0x35 ); // Help request unchanged.
			}
			else if ( index == 1 )
			{
				PageEntry entry = PageQueue.GetEntry( m_From );

				if ( entry != null && entry.Handler == null )
				{
					m_From.SendLocalizedMessage( 1005307, "", 0x35 ); // Removed help request.
					PageQueue.Remove( entry );
				}
				else
				{
					m_From.SendLocalizedMessage( 1005306, "", 0x35 ); // Help request unchanged.
				}
			}
		}
	}

	public class HelpGump : Gump
	{
		public static void Initialize()
		{
			EventSink.HelpRequest += new HelpRequestEventHandler( EventSink_HelpRequest );
		}

		private static void EventSink_HelpRequest( HelpRequestEventArgs e )
		{
			foreach ( Gump g in e.Mobile.NetState.Gumps )
			{
				if ( g is HelpGump )
					return;
			}

			if ( !PageQueue.CheckAllowedToPage( e.Mobile ) )
				return;

			if ( PageQueue.Contains( e.Mobile ) )
				e.Mobile.SendMenu( new ContainedMenu( e.Mobile ) );
			else
				e.Mobile.SendGump( new HelpGump( e.Mobile ) );
		}

		private static bool IsYoung( Mobile m )
		{
			if ( m is PlayerMobile )
				return ((PlayerMobile)m).Young;

			return false;
		}

		public static bool CheckCombat( Mobile m )
		{
			for ( int i = 0; i < m.Aggressed.Count; ++i )
			{
				AggressorInfo info = m.Aggressed[i];

				if ( DateTime.Now - info.LastCombatTime < TimeSpan.FromSeconds( 30.0 ) )
					return true;
			}

			return false;
		}

		public HelpGump( Mobile from ) : base( 0, 0 )
		{
			from.CloseGump( typeof( HelpGump ) );

			AddBackground( 50, 25, 540, 430, 2600 );

			AddPage( 0 );

			AddHtmlLocalized( 150, 50, 360, 40, 1001002, false, false ); // <CENTER><U>Ultima Online Help Menu</U></CENTER>
			AddButton( 425, 415, 2073, 2072, 0, GumpButtonType.Reply, 0 ); // Close

			AddPage( 1 );

			AddButton( 80, 90, 5540, 5541, 1, GumpButtonType.Reply, 2 );
			AddHtml( 110, 90, 450, 74, @"<u>General question about Ultima Online.</u> Select this option if you have a general gameplay question, need help learning to use a skill, or if you would like to search the UO Knowledge Base.", true, true );

			AddButton( 80, 170, 5540, 5541, 2, GumpButtonType.Reply, 0 );
			AddHtml( 110, 170, 450, 74, @"<u>My character is physically stuck in the game.</u> This choice only covers cases where your character is physically stuck in a location they cannot move out of. This option will only work two times in 24 hours.", true, true );
		}

		public override void OnResponse( NetState state, RelayInfo info )
		{
			Mobile from = state.Mobile;

			PageType type = (PageType)(-1);

			switch ( info.ButtonID )
			{
				case 0: // Close/Cancel
				{
					from.SendLocalizedMessage( 501235, "", 0x35 ); // Help request aborted.

					break;
				}
				case 1: // General question
				{
					type = PageType.Question;
					break;
				}
				case 2: // Stuck
				{
					BaseHouse house = BaseHouse.FindHouseAt( from );

					if ( house != null && house.IsAosRules )
					{
						from.Location = house.BanLocation;
					}
					else if ( from.Region.IsPartOf( typeof( Server.Regions.Jail ) ) )
					{
						from.SendLocalizedMessage( 1041530, "", 0x35 ); // You'll need a better jailbreak plan then that!
					}
					else if ( from.CanUseStuckMenu() && from.Region.CanUseStuckMenu( from ) && !CheckCombat( from ) && !from.Frozen && !from.Criminal && (Core.AOS || from.Kills < 5) )
					{
						StuckMenu menu = new StuckMenu( from, from, true );

						menu.BeginClose();

						from.SendGump( menu );
					}
					else
					{
						type = PageType.Stuck;
					}

					break;
				}
				case 3: // Report bug or contact Origin
				{
					type = PageType.Bug;
					break;
				}
				case 4: // Game suggestion
				{
					type = PageType.Suggestion;
					break;
				}
				case 5: // Account management
				{
					type = PageType.Account;
					break;
				}
				case 6: // Other
				{
					type = PageType.Other;
					break;
				}
				case 7: // Harassment: verbal/exploit
				{
					type = PageType.VerbalHarassment;
					break;
				}
				case 8: // Harassment: physical
				{
					type = PageType.PhysicalHarassment;
					break;
				}
				case 9: // Young player transport
				{
					if ( IsYoung( from ) )
					{
						if ( from.Region.IsPartOf( typeof( Regions.Jail ) ) )
						{
							from.SendLocalizedMessage( 1041530, "", 0x35 ); // You'll need a better jailbreak plan then that!
						}
						else if ( from.Region.IsPartOf( "Haven Island" ) )
						{
							from.SendLocalizedMessage( 1041529 ); // You're already in Haven
						}
						else
						{
							from.MoveToWorld( new Point3D( 3503, 2574, 14 ), Map.Felucca );
						}
					}

					break;
				}
			}

			if ( type != (PageType)(-1) && PageQueue.CheckAllowedToPage( from ) )
				from.SendGump( new PagePromptGump( from, type ) );
		}
	}
}